/*******************************************************************************************
 Demonstrate transmission of RGB data to a chain of WS2811 LED string drivers.
*
*  SCT1_OUT0 is the transmit data line.
*  Transmission speed is 800 kBit/s, but can be adjusted down to 400 kHz as well.
*  full data bit time   FB  = 1.25 us at 800kb/s
*  data bit 0 high time T0H = 0.25 us at 800kb/s (is 20% of full bit time)
*  data bit 1 high time T1H = 0.6  us at 800kb/s (is 48% of full bit time)
*******************************************************************************************/
#include "LPC15xx.h"                                       // LPC15xx definitions
#include "sct1_ws2811.h"

#define DATA_OUT          0                               // use SCT1_OUT0 as data output
#define AUX_OUT           1                               // use SCT1_OUT1 as aux output
#define DATA_SPEED        800000

#define WS2811_FB         (SystemCoreClock/DATA_SPEED)    // full data bit time   = 90 (at 800 kb/s)
#define WS2811_T0H        ((WS2811_FB*20)/100)            // data bit 0 high time = 18 (at 800 kb/s)
#define WS2811_T1H        ((WS2811_FB*48)/100)            // data bit 1 high time ~ 43 (at 800 kb/s)

void LEDDRIVER_open(void)
{
    LPC_SYSCON->SYSAHBCLKCTRL1 |= EN1_SCT1;                // enable the SCT1 clock

    LPC_SCT1->STATE_L         = 12;                        // start state
    LPC_SCT1->OUTPUT         &= ~((1 << DATA_OUT)          // preset data output low
                              |   (1 << AUX_OUT));         // preset aux output low

    LPC_SCT1->MATCHREL_L[0]   = WS2811_FB - 1;             // full bit period
    LPC_SCT1->MATCHREL_L[1]   = WS2811_T0H - 1;            // T0H ticks
    LPC_SCT1->MATCHREL_L[2]   = WS2811_T1H - 1;            // T1H ticks

    LPC_SCT1->EVENT[15].STATE = 0xFFFFFFFF;                // happens in all states
    LPC_SCT1->EVENT[15].CTRL  = (0  << 0)                  // related to MATCH0_L
                              | (1  << 12)                 // match only
                              | (0  << 14)                 // add value to STATE
                              | (31 << 15);                // add 31 (i.e subtract 1)

    LPC_SCT1->EVENT[14].STATE = 0x00000FFE;                // all data bit states except state 0
    LPC_SCT1->EVENT[14].CTRL  = (2  << 0)                  // related to MATCH2_L
                              | (1  << 12);                // match only

    LPC_SCT1->EVENT[13].STATE = 0x00000FFF;                // all data bit states
    LPC_SCT1->EVENT[13].CTRL  = (1  << 0)                  // related to MATCH1_L
                              | (1  << 12);                // match only

    LPC_SCT1->EVENT[12].STATE = 0;                         // contains the 12 bits written by the app
    LPC_SCT1->EVENT[12].CTRL  = (1  << 0)                  // related to MATCH1_L
                              | (1  << 5)                  // use OUTPUT for I/O condition
                              | (AUX_OUT << 6)             // Use AUX output signal
                              | (0  << 10)                 // AUX = 0
                              | (3  << 12);                // match AND I/O

    LPC_SCT1->EVENT[11].STATE = 0;                         // contains the 12 bits written by the app
    LPC_SCT1->EVENT[11].CTRL  = (1  << 0)                  // related to MATCH1_L
                              | (1  << 5)                  // use OUTPUT for I/O condition
                              | (AUX_OUT << 6)             // use AUX output signal
                              | (3  << 10)                 // AUX = 1
                              | (3  << 12);                // match AND I/O

    LPC_SCT1->EVENT[10].STATE = 0x00000001;                // happens only in state 0
    LPC_SCT1->EVENT[10].CTRL  = (2  << 0)                  // related to MATCH2_L
                              | (1  << 12)                 // match only
                              | (1  << 14)                 // set STATE to a value
                              | (12 << 15);                // set STATE to 12

    LPC_SCT1->OUT[AUX_OUT].SET  = (1 << 10);               // Event 10 toggles the AUX signal
    LPC_SCT1->OUT[AUX_OUT].CLR  = (1 << 10);               // Event 10 toggles the AUX signal

    LPC_SCT1->OUT[DATA_OUT].SET = (1 << 15)                // Event 15 sets the DATA signal
                                   | (1 << 12)             // Event 12 sets the DATA signal
                                   | (1 << 11);            // Event 11 sets the DATA signal
    LPC_SCT1->OUT[DATA_OUT].CLR = (1 << 14)                // Event 14 clears the DATA signal
                                   | (1 << 13)             // Event 13 clears the DATA signal
                                   | (1 << 10);            // Event 10 clears the DATA signal

    LPC_SCT1->RES                  = (0 << 2 * DATA_OUT)   // on conflict: DATA signal doesn't change
                                   | (3 << 2 * AUX_OUT);   // on conflict: AUX signal toggles

    LPC_SCT1->LIMIT_L = (1 << 15);                         // Event 15 limits the counter
    LPC_SCT1->EVFLAG  = (1 << 10);                         // clear event 10 irq flag
    LPC_SCT1->EVEN   |= (1 << 10);                         // Event 10 generates an irq

    NVIC_EnableIRQ(SCT1_IRQn);                             // enable SCT1 interrupt
}

void LEDDRIVER_write(uint32_t rgb)                         // function to write to a transmit buffer
{
    if (LPC_SCT1->OUTPUT & (1 << AUX_OUT))                 // aux output determines which buffer to write
    {
        LPC_SCT1->EVENT[12].STATE = rgb & 0xFFF;
    }
    else
    {
        LPC_SCT1->EVENT[11].STATE = rgb & 0xFFF;
    }
}

void LEDDRIVER_haltAfterFrame(int on)                      // (de)activate HALT after next frame
{
    LPC_SCT1->HALT_L = (on << 10);                         // if last, event 10 halts the counter
}

void LEDDRIVER_start(void)                                 // start a frame transmission
{
    LPC_SCT1->COUNT_L = - WS2811_FB * 50;                  // guarantee 50 µs min time between transfers
    LPC_SCT1->STATE_L = 0;                                 // start state
    LPC_SCT1->CTRL_L &= ~(1 << 2);                         // start timer H
}
